
//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/ 
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License. 
//
// The Original Code is: all this file. 
//
// The Initial Developer of the Original Code is
// Trevor Nash of Melvaig Software Engineering Limited (tcn@melvaig.co.uk).
//
// Portions created by (your name) are Copyright (C) (your legal entity). All Rights Reserved. 
//
// Contributor(s): none. 
//
package com.icl.saxon.resolvers;
import com.icl.saxon.OutputURIResolver;
import java.lang.String;
import java.lang.System;
import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.TransformerException;
import javax.xml.transform.URIResolver;

public class InternalURIResolver implements OutputURIResolver, URIResolver {

    private boolean trace = false;
    public void setTrace (boolean value) {
        trace = value;
    }

    private Hashtable map = new Hashtable();

    public Map getMap() {
        return map;
    }


    /**
    * Associate a Source object with a given URI.  The document supplied
    * may be read with xsl:include, xsl:import or the document() function.
    */
    public void putSource(
        String uri,
        Source value) {
        map.put(uri, value);
    }

    /**
    * Retrieve a Source object corresponding to a given URI.  This is
    * used to retrieve a value written with putSource.
    */
    public Source getSource(
        String uri) {
        Object result = map.get(uri);
        if (result instanceof Source) {
            return (Source)result;
        }
        return null;
    }

    /**
    * Associate a Result object with a given URI.  The document supplied
    * may be written with xsl:document.
    */
    public void putResult(
        String uri,
        Result value) {
        map.put(uri, value);
    }

    /**
    * Create a Result object with a given URI.  The document supplied
    * may be written with xsl:document and read in a subsequent
    * transformation with xsl:include, xsl:import or the document() function.
    */
    public void putResult(String uri) {
        DOMResult dom = new DOMResult();
        dom.setSystemId(uri);
        map.put(uri, dom);
    }

    /**
    * Retrieve a Result object corresponding to a given URI.  This is
    * used to access a document written with xsl:document.
    */
    public Result getResult(
        String uri) {
        Object result = map.get(uri);
        if (result instanceof Result) {
            return (Result)result;
        }
        return null;
    }

    /**
    * Retrieve a Result object corresponding to a given URI and
    * convert it to a Source.  This is
    * used to access a document written with xsl:document for
    * processing in another transformation.  This only works for
    * a DOMResult, as other types of result cannot be converted.
    */
    public Source getResultAsSource(
        String uri) {
        Object result = map.get(uri);
        if (result instanceof DOMResult) {
            DOMResult dom = (DOMResult) result;
            return new DOMSource(dom.getNode(), dom.getSystemId());
        }
        return null;
    }


    public InternalURIResolver () {}

    
    /**
     * Called by the processor when it encounters
     * an xsl:document element.
     *
     * @param href An href attribute, which may be relative or absolute.
     * @param base The base URI in effect when the href attribute
     * was encountered.
     * @param attributes The attributes of the xsl:document element, after
     * AVT expansion.
     *
     * @return A Result object, or null if the href cannot be resolved,
     * and the processor should try to resolve the URI itself.
     *
     * @throws TransformerException if an error occurs when trying to
     * resolve the URI.
     */
    public Result resolveNew (
            String href,
            String base,
            Properties attributes)
        throws TransformerException
    {
        // If there is a previously registered result with this URI, use that.
        Result result = getResult(href);
        if (result != null) {
            if (trace) System.out.println("Output URI resolved: "+href);

            return result;
        }
        // If the URI is not marked 'internal:' then let the standard resolver
        // handle it.
        if (!href.startsWith("internal:")) {
            return null;
        }
        // For 'internal:' URIs create a DOMResult (which can subsequently be
        // converted to a Source).
        DOMResult dom = new DOMResult();
        dom.setSystemId (href);
        map.put(href, dom);
        if (trace) System.out.println("Output URI created: "+href);

        return dom;
    }


    
    /**
    * Called by the processor when it encounters an xsl:include, xsl:import,
    * or document() function.
    * @param href An href attribute, which may be relative or absolute.
    * @params base The base URI in effect when the href attribute was encountered.
    * @return A Source object, or null if the href cannot be resolved, and the
    * processor should try to resolve the URI itself.
    */
    public Source resolve (
            String href,
            String base)
        throws TransformerException
    {
        // First try for a Source object.
        Source result = getSource(href);
        if (result != null) {
            if (trace) System.out.println("URI resolved: "+href);

            return result;
        }
        // Then a Result object which can be converted.
        result = getResultAsSource(href);
        if (result != null) {
            if (trace) System.out.println("URI resolved from output: "+href);

            return result;
        }
        // That failing, give up and let the standard resolver deal with it.
        if (trace) System.out.println("URI not resolved: "+href);

        return null;
    }


}
       