<?xml version="1.0" encoding="utf-8"?>
<xsl:transform xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:m21="http://melvaig.com/m21" xmlns:saxon="http://icl.com/saxon" xmlns:lxsl="http://melvaig.com/m21/literal-xsl" version="1.1">

<xsl:key name="node-by-id" match="*" use="@id"/>
<saxon:function name="m21:object-reference">
   <xsl:param name="type"/>
   <xsl:param name="id"/>
   <xsl:for-each select="$root">
      <xsl:variable name="return" select="key('node-by-id', $id)[name()=$type]"/>
      <xsl:if test="count($return)!=1">
         <xsl:message>
            <xsl:text>m21:object-reference id "</xsl:text>
            <xsl:value-of select="$id"/>
            <xsl:text>" identifies </xsl:text>
            <xsl:value-of select="count($return)"/>
            <xsl:text> elements</xsl:text>
         </xsl:message>
      </xsl:if>
      <saxon:return select="$return"/>
   </xsl:for-each>
</saxon:function>
   
<xsl:output method="text"/>
   
<xsl:variable name="root" select="/"/>
   
<!-- This is the default search order for template files; look relative
     to the file containing the m21:file element first, then the root
     document and lastly the stylesheet, i.e. the m21 package.
     Note that because the second argument to document() is a node set
     and the base URI is the base URI of the first node rather than its
     value, it is not possible to specify arbitrary locations here.
     It is of course possible to specify an absolute URI to the type
     or template attributes of m21:file.
     A later version of M21 may allow other
     schemes implemented via a custom URI resolver.
     23.1.2002: source withdrawn because we do not have access
     to nodes from the original input document(s).  The 'm21' method
     will look in the current directory as well (we expect that to be
     where the main input document is).
  -->
<xsl:variable name="search-path" select="'root m21'"/>
   
<saxon:function name="m21:id-list">
   <xsl:param name="list"/>
   <xsl:choose>
      <xsl:when test="$list">
         <saxon:return select="concat(              concat($list[1]/@id, ' '),              m21:id-list($list[position()!=1]))"/>
      </xsl:when>
      <xsl:otherwise>
         <saxon:return select="''"/>
      </xsl:otherwise>
   </xsl:choose>
</saxon:function>
   
<!-- m21:sources with a file.id : combine all combinations of sink
     and file, and if file.id is _parent take the file.ids of the nearest
     ancestor.
     without a file.id: all combinations of sink with all m21:file/@id
  -->
<xsl:key name="source_by_ids" match="m21:source" use="m21:source-keys(., @sink.id)"/>
<saxon:function name="m21:source-keys">
   <xsl:param name="source"/>
   <xsl:param name="sink.id"/>
   <!-- optimisation: common case where file and sink ids are not lists -->
   <xsl:variable name="simple-key" select="concat($source/@file.id, concat('::', $sink.id))"/>
   <xsl:choose>
      <xsl:when test="$source/@file.id='_parent'">
         <saxon:return select="m21:source-keys($source/ancestor::m21:source[1], $sink.id)"/>
      </xsl:when>
      <xsl:when test="$source/@file.id and not(contains($simple-key, ' '))">
         <saxon:return select="$simple-key"/>
      </xsl:when>
      <xsl:otherwise>
         <!--putting this 'optimisation' in and using the variable in
             the inner for-each actually puts the run time up from 14.1
             to 15.2.  (before simple-key optimisation above) -->
         <!--xsl:variable name="sink-ids" select="saxon:tokenize($sink.id)" /-->
         <xsl:variable name="keys">
            <xsl:for-each select="saxon:if(                        $source/@file.id,                        saxon:tokenize($source/@file.id),                        //m21:file/@id)">
               <xsl:variable name="major" select="concat(' ', concat(., '::'))"/>
               <xsl:for-each select="saxon:tokenize($sink.id)">
                  <xsl:value-of select="concat($major, .)"/>
               </xsl:for-each>
            </xsl:for-each>
         </xsl:variable>
         <saxon:return select="saxon:tokenize($keys)"/>
      </xsl:otherwise>
   </xsl:choose>
</saxon:function>
   
<xsl:param name="out-dir" select="''"/>
   

<xsl:key name="file-by-id" match="m21:file" use="@id"/>
<xsl:key name="file-by-id" match="m21:file" use="'_all_'"/>
<saxon:function name="m21:file-by-id">
   <xsl:param name="id"/>
   <xsl:for-each select="$generated">
      <xsl:variable name="return" select="key('file-by-id', $id)"/>
      <xsl:if test="count($return)!=1">
         <xsl:message>
            <xsl:text>m21:file id "</xsl:text>
            <xsl:value-of select="$id"/>
            <xsl:text>" identifies </xsl:text>
            <xsl:value-of select="count($return)"/>
            <xsl:text> m21:file elements</xsl:text>
         </xsl:message>
      </xsl:if>
      <saxon:return select="$return"/>
   </xsl:for-each>
</saxon:function>

<xsl:variable name="generated">
   <xsl:apply-templates select="/*" mode="generate"/>
</xsl:variable>
<xsl:template match="*" mode="generate" priority="-1">
    <xsl:element name="{name()}" namespace="{namespace-uri()}">
        <xsl:copy-of select="@*"/>
        <xsl:apply-templates mode="generate"/>
    </xsl:element>
</xsl:template>
<xsl:template match="comment()|processing-instruction()" mode="generate" priority="-1">
   <xsl:copy/>
</xsl:template>
<xsl:template match="/">
   <xsl:if xmlns:m21p="java:com.melvaig.m21.M21Processor" test="m21p:trace()" m21p:dummy="">
      <xsl:document href="internal:_generated" method="xml" indent="yes">
         <xsl:copy-of select="$generated"/>
      </xsl:document>
   </xsl:if>
   <xsl:for-each select="$generated">
      <xsl:apply-templates select="key('file-by-id', '_all_')"/>
   </xsl:for-each>
</xsl:template>
   
<!-- This is the m21:file from the main document where the skeleton
     is given in a separate file
  -->
<xsl:template match="m21:file[@type|@template]">
   <xsl:variable name="uri" select="saxon:if(@template, @template, concat(@type,'.xml'))"/>
   <xsl:variable name="doc" select="m21:find-document($uri, m21:file-by-id(@id))"/>
   <xsl:if test="$doc/*">
      <xsl:call-template name="generate-file">
         <xsl:with-param name="file" select="."/>
         <xsl:with-param name="document" select="$doc"/>
      </xsl:call-template>
   </xsl:if>
   <xsl:if test="not($doc/*)">
      <xsl:message>
         <xsl:value-of select="name()"/>/@id='<xsl:value-of select="@id"/>
         <xsl:text>' Cannot find </xsl:text>
         <xsl:value-of select="$uri"/>
         <xsl:text> &#x0A;</xsl:text>
         <!-- we cannot use saxon:systemId() here because the m21:node is a copy
              of the original; it might even be generated from an object -->
      </xsl:message>
   </xsl:if>
</xsl:template>
   
<!-- This is the m21:file from the main document which has an embedded
     skeleton
  -->
<xsl:template match="m21:file">
   <xsl:call-template name="generate-file">
      <xsl:with-param name="file" select="."/>
      <xsl:with-param name="document" select="."/>
   </xsl:call-template>
</xsl:template>
   
<!-- given a file specified by a relative URI, return a document found
     at one of the locations specified by the global variable
     $search-path.
  -->
<saxon:function name="m21:find-document-in">
   <xsl:param name="name"/>
   <xsl:param name="file-node"/>
   <xsl:param name="look-in"/>
   <xsl:choose>
      <xsl:when test="$look-in='source'">
         <saxon:return select="document($name, $file-node)"/>
      </xsl:when>
      <xsl:when test="$look-in='root'">
         <saxon:return select="document($name, $root)"/>
      </xsl:when>
      <xsl:when test="$look-in='m21'">
         <saxon:return select="document($name)"/>
      </xsl:when>
   </xsl:choose>
</saxon:function>

<saxon:function name="m21:find-document">
   <xsl:param name="name"/>
   <xsl:param name="file-node"/>
   <xsl:param name="path" select="$search-path"/>
   <xsl:variable name="npath" select="concat(normalize-space($path), ' ')"/>
   <xsl:variable name="look-in" select="substring-before($npath, ' ')"/>
   <xsl:variable name="rest-of-path" select="substring-after($npath, ' ')"/>
   <xsl:variable name="doc" select="m21:find-document-in($name, $file-node, $look-in)"/>
   <xsl:choose>
      <xsl:when test="$doc/node()">     <!-- doc has been found -->
         <saxon:return select="$doc"/>  <!-- return it -->
      </xsl:when>
      <xsl:when test="$rest-of-path">    <!-- somewhere else to look -->
                                         <!-- recursive call -->
         <saxon:return select="m21:find-document($name, $file-node, $rest-of-path)"/>
      </xsl:when>
      <xsl:otherwise>                    <!-- no more places to look -->
         <saxon:return select="$file-node"/>
                                         <!-- return empty document -->
      </xsl:otherwise>
   </xsl:choose>
</saxon:function>
   
<!-- A m21:sink in a referenced file.xml: go back to the main document to find
     all the m21:source elements with the same sink.id.
     If the m21:sink has a body, then this is copied for each m21:source; see
     also m21:copy.
  -->
<xsl:template match="m21:sink" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:variable name="sink" select="."/>
   <xsl:for-each select="$generated"> <!-- context for key() -->
      <xsl:apply-templates select="&#xA;        key('source_by_ids', concat($file.id, concat('::', $sink/@id)))&#xA;      " mode="writing">
         <xsl:with-param name="file.id" select="$file.id"/>
         <xsl:with-param name="sink" select="$sink"/>
         <xsl:with-param name="outer" select="true()"/>
      </xsl:apply-templates>
   </xsl:for-each>
</xsl:template>
   
<!-- A m21:sorted-sink in a referenced file.xml: just like m21:sink except
     that the sources are inserted in key sequence.  The sort order
     is the textual content of the m21:source element, i.e. string(.),
     or the value of the 'key' attribute if present.
     'order' and 'data-type' attributes are available as defined on xsl:sort.
     Other sort orders will be allowed in a future release by the
     'generic template' mechanism and/or by generating a style sheet.
  -->
<xsl:template match="m21:sorted-sink" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:variable name="sink" select="."/>
   <xsl:for-each select="$generated"> <!-- context for key() -->
      <xsl:apply-templates select="&#xA;        key('source_by_ids', concat($file.id, concat('::', $sink/@id)))&#xA;      " mode="writing">
         <xsl:sort select="saxon:if(@key, @key, .)" order="{saxon:if($sink/@order, $sink/@order, 'ascending')}" data-type="{saxon:if($sink/@data-type, $sink/@data-type, 'text')}"/>
         <xsl:with-param name="file.id" select="$file.id"/>
         <xsl:with-param name="sink" select="$sink"/>
         <xsl:with-param name="outer" select="true()"/>
      </xsl:apply-templates>
   </xsl:for-each>
</xsl:template>
   
<!-- A m21:sorted-sink in a referenced file.xml qualified by select="unique":
     just like m21:sorted-sink except that if more than one source with the
     same key exists, only the first is used.
  -->
<xsl:template match="m21:sorted-sink[@select='unique']" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:variable name="sink" select="."/>
   <xsl:for-each select="$generated"> <!-- context for key() -->
      <xsl:variable name="source-nodes" select="&#xA;        key('source_by_ids', concat($file.id, concat('::', $sink/@id)))&#xA;      "/>
      <xsl:apply-templates select="saxon:distinct(                         $source-nodes,                         saxon:expression('saxon:if(@key, @key, .)'))" mode="writing">
         <xsl:sort select="saxon:if(@key, @key, .)" order="{saxon:if($sink/@order, $sink/@order, 'ascending')}" data-type="{saxon:if($sink/@data-type, $sink/@data-type, 'text')}"/>
         <xsl:with-param name="file.id" select="$file.id"/>
         <xsl:with-param name="sink" select="$sink"/>
         <xsl:with-param name="outer" select="true()"/>
      </xsl:apply-templates>
   </xsl:for-each>
</xsl:template>
   
<xsl:template match="m21:source[@file.id]" mode="writing-attributes">
   <xsl:param name="file.id"/>
   <xsl:apply-templates select="*" mode="writing-attributes">
      <xsl:with-param name="file.id" select="$file.id"/>
   </xsl:apply-templates>
</xsl:template>
   
<!-- A m21:source element for a matching m21:sink not qualified by
     a file.id.
  -->
<xsl:template match="m21:source" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:param name="sink"/>
   <xsl:param name="outer" select="false()"/>
   <xsl:if test="$outer">
      <xsl:choose>
         <xsl:when test="$sink/node()">
            <!-- m21:sink has a body, so use this as a template for inserting
                 the source
            -->
            <xsl:apply-templates select="$sink/node()" mode="writing">
               <xsl:with-param name="file.id" select="$file.id"/>
               <xsl:with-param name="sink" select="$sink"/>
               <xsl:with-param name="source" select="."/>
               <xsl:with-param name="position" select="position()"/>
               <xsl:with-param name="last" select="last()"/>
            </xsl:apply-templates>
         </xsl:when>
         <xsl:otherwise>
            <xsl:apply-templates mode="writing">
               <xsl:with-param name="file.id" select="$file.id"/>
               <xsl:with-param name="sink" select="$sink"/>
               <xsl:with-param name="source" select="."/>
            </xsl:apply-templates>
         </xsl:otherwise>
      </xsl:choose>
   </xsl:if>
</xsl:template>

<xsl:template match="m21:source" mode="writing-attributes">
   <xsl:param name="file.id"/>
   <xsl:apply-templates select="*" mode="writing-attributes">
      <xsl:with-param name="file.id" select="$file.id"/>
   </xsl:apply-templates>
</xsl:template>
   
<xsl:template match="m21:copy" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:param name="sink"/>
   <xsl:param name="source" select="."/>
   <xsl:param name="position" select="0"/>
   <xsl:param name="last" select="0"/>
   <xsl:apply-templates select="$source/node()" mode="writing">
       <xsl:with-param name="file.id" select="$file.id"/>
       <xsl:with-param name="sink" select="$sink"/>
       <!-- fix the $source parameter so that we can detect loops -->
       <xsl:with-param name="source" select="."/>
       <xsl:with-param name="position" select="$position"/>
       <xsl:with-param name="last" select="$last"/>
   </xsl:apply-templates>
</xsl:template>
   
<xsl:template match="m21:separator" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:param name="sink"/>
   <xsl:param name="source" select="."/>
   <xsl:param name="position" select="0"/>
   <xsl:param name="last" select="0"/>
   <xsl:if test="           (not(@test)) and ($position != 1) or           (@test='first') and ($position = 1) or           (@test='not first') and ($position != 1) or           (@test='last') and ($position = $last) or           (@test='not last') and ($position != $last)         ">
      <xsl:value-of select="@value"/>
      <xsl:apply-templates mode="writing">
         <xsl:with-param name="file.id" select="$file.id"/>
         <xsl:with-param name="sink" select="$sink"/>
         <xsl:with-param name="source" select="$source"/>
         <xsl:with-param name="position" select="$position"/>
         <xsl:with-param name="last" select="$last"/>
      </xsl:apply-templates>
   </xsl:if>
</xsl:template>
   
<xsl:template name="generate-file">
   <xsl:param name="file"/>
   <xsl:param name="document"/>
   <xsl:variable name="method" select="saxon:if($file/@method, $file/@method,saxon:if($document/m21:text-file, 'text',           saxon:if($document/html, 'html', 'xml')))"/>

   <xsl:value-of select="$file/@id"/><xsl:text>: method=</xsl:text>
   <xsl:value-of select="$method"/>
   <xsl:text>, file='</xsl:text>
   <xsl:value-of select="$file/@name"/>
   <xsl:text>'&#x0A;</xsl:text>

   <xsl:document href="{$out-dir}{$file/@name}" method="{$method}">
      <xsl:if test="$method='xml'">
         <!-- for XML, put out a newline to put the XML
              declaration on a line by itself
           -->
         <xsl:text>&#x0A;</xsl:text>
      </xsl:if>
      <xsl:apply-templates select="$document/node()" mode="writing">
         <xsl:with-param name="file.id" select="$file/@id"/>
      </xsl:apply-templates>
   </xsl:document>
</xsl:template>
   
<!-- When the output method is XML the default template for elements
     will not do: we want to copy the element itself, not just
     the children.
  -->
<xsl:template match="*" mode="writing" priority="-2">
   <xsl:param name="file.id"/>
   <xsl:param name="sink" select="/.."/>
   <xsl:param name="source" select="/.."/>
   <!-- use xsl:element instead of xsl:copy to avoid copying
        superfluous namespaces e.g. m21.
        The namespace is specified explicitly in case the prefix in the
        document is bound to a different namespace in the stylesheet,
        or maybe not bound at all.
        Re. the method from XSLT programmer's reference, Ed1 pg 488
        (description of name() function) which recommends local-name() -
        the output document is correct, but the prefix is gratuitously
        removed.
     -->
   <xsl:element name="{name()}" namespace="{namespace-uri()}">
      <xsl:apply-templates select="@*" mode="writing">
         <xsl:with-param name="file.id" select="$file.id"/>
         <xsl:with-param name="sink" select="$sink"/>
         <xsl:with-param name="source" select="$source"/>
      </xsl:apply-templates>
      <xsl:apply-templates mode="writing">
         <xsl:with-param name="file.id" select="$file.id"/>
         <xsl:with-param name="sink" select="$sink"/>
         <xsl:with-param name="source" select="$source"/>
      </xsl:apply-templates>
   </xsl:element>
</xsl:template>
   
<xsl:template match="lxsl:*" mode="writing" priority="-1">
   <xsl:param name="file.id"/>
   <xsl:param name="sink"/>
   <xsl:param name="source"/>
   <xsl:element name="{concat('xsl:', local-name())}" namespace="http://www.w3.org/1999/XSL/Transform">
      <xsl:apply-templates select="@*|node()" mode="writing">
         <xsl:with-param name="file.id" select="$file.id"/>
         <xsl:with-param name="sink" select="$sink"/>
         <xsl:with-param name="source" select="$source"/>
      </xsl:apply-templates>
   </xsl:element>
</xsl:template>
   
<xsl:template match="comment()|processing-instruction()" mode="writing" priority="-1">
   <xsl:copy/>
</xsl:template>
   
<xsl:template match="@m21:sink" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:variable name="sink.id" select="."/>
   <xsl:for-each select="$generated"> <!-- context for key() -->
      <xsl:apply-templates select="         key('source_by_ids', concat($file.id, concat('::', $sink.id)))" mode="writing-attributes">
         <xsl:with-param name="file.id" select="$file.id"/>
      </xsl:apply-templates>
   </xsl:for-each>
</xsl:template>
   
<xsl:template match="@*" mode="writing">
   <xsl:copy/>
</xsl:template>
   
<!-- m21:verbatim is used to preserve a particular character representation.
     The main purpose is to generate XSL files with instructions like
     <xsl:text>&#0A;</xsl:text>.  Without this facility the XSL would end
     up with a real newline: the semantics are the same but some people prefer
     to see the &#x0A; preserved.
     Note that to get this to work properly the content of the element should
     be a <![CDATA[...]]>.
  -->
<xsl:template match="m21:verbatim" mode="writing">
   <xsl:value-of select="." disable-output-escaping="yes"/>
</xsl:template>
   
<xsl:template match="m21:namespace" mode="writing-attributes">
   <xsl:variable name="prefix" select="@prefix"/>
   <xsl:copy-of select="namespace::*[local-name() = $prefix]"/>
</xsl:template>
<!-- preserve namespace nodes on this element, or we will not
     be able to copy them -->
<xsl:template match="m21:namespace" mode="generate">
    <xsl:copy>
        <xsl:copy-of select="@*"/>
        <xsl:apply-templates mode="generate"/>
    </xsl:copy>
</xsl:template>
   
<xsl:template match="m21:attribute" mode="writing-attributes">
   <xsl:attribute name="{@name}">
      <xsl:apply-templates mode="writing-attributes"/>
   </xsl:attribute>
</xsl:template>
   
<!-- m21:case element from a matching m21:source element whose text is
     required.
  -->
<xsl:template match="m21:case" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:if test="@file.id=$file.id"><xsl:apply-templates/></xsl:if>
</xsl:template>
   
<!-- A m21:id in a referenced file.xml: insert the file identifier
     (the id attribute of the m21:file element)
  -->
<xsl:template match="m21:id" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:value-of select="$file.id"/>
</xsl:template>
   
<xsl:template match="m21:copy-marked-section" mode="writing">
   <xsl:param name="file.id"/>
   <xsl:variable name="file-element" select="m21:file-by-id(@file.id)"/>
   <xsl:if test="count($file-element)!=1">
      <xsl:text>Found </xsl:text><xsl:value-of select="count($file-element)"/>
      <xsl:text> m21:file elements with id = '</xsl:text><xsl:value-of select="@file.id"/><xsl:text>'</xsl:text>
      <xsl:text> &#x0A;</xsl:text>
      <xsl:value-of select="saxon:systemId()"/>
      <xsl:text> line </xsl:text>
      <xsl:value-of select="saxon:lineNumber()"/>
   </xsl:if>
   <!-- also should check a marked section with the given ID exists. -->
   <xsl:choose>
      <xsl:when test="$file-element/@type">
         <xsl:variable name="doc" select="m21:find-document(concat($file-element/@type,'.xml'), $file-element)"/>
         <xsl:apply-templates select="$doc//m21:marked-section[@id=current()/@marked-section.id]/node()" mode="writing">
            <xsl:with-param name="file.id" select="$file.id"/>
         </xsl:apply-templates>
      </xsl:when>
      <xsl:when test="$file-element/@template">
         <xsl:variable name="doc" select="m21:find-document($file-element/@template, $file-element)"/>
         <xsl:apply-templates select="$doc//m21:marked-section[@id=current()/@marked-section.id]/node()" mode="writing">
            <xsl:with-param name="file.id" select="$file.id"/>
         </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
         <xsl:apply-templates select="$file-element//m21:marked-section[@id=current()/@marked-section.id]/node()" mode="writing">
            <xsl:with-param name="file.id" select="$file.id"/>
         </xsl:apply-templates>
      </xsl:otherwise>
   </xsl:choose>
</xsl:template>
   
<xsl:template match="m21:marked-section" mode="writing">
</xsl:template>
   

</xsl:transform>